# -*- coding: utf-8 -*-

"""
The postman project - config.py

This file is released under the BSD license, see LICENSE for
more information.

Francisco de Borja Lopez Rio - <borja@codigo23.net>
Soluciones Informaticas Codigo23 S.L.U. - http://codigo23.net
"""

import os
from ConfigParser import SafeConfigParser

class DaemonConfig():
    def __init__(self, configfile=None):
        self.configfile = configfile
        self.default_paths = ['/usr/local/etc/postman', '/usr/local/etc',
                              '/etc/postman', '/etc',
                              os.path.join(os.path.dirname(__file__),'../conf')]
        self.config = {}

    @property
    def default_filename(self):
        """
        Method to get the default configuration filename. It is a method instead
        of an attribute for easier subclassing
        """
        return 'postman.conf'

    @property
    def sections(self):
        """
        Method to get the list of available sections in the configuration file.
        It is a method instead of an attribute for easier subclassing.

        FIXME: Perhaps it would be better to dinamically get the list of
        sections from the configuration file itself.
        """
        return ['xmlrpc_server', 'storage', 'archive',
                'mailing_lists', 'members']
    
    def validate_configfile(self):        
        if self.configfile:
            if not os.path.exists(self.configfile):
                # the file does not exist, so we override it with None,
                # allowing the next few lines to search for a valid
                # config file on the usual places
                self.configfile = None
            
        if not self.configfile:
            for path in self.default_paths:
                full_path = os.path.join(path, self.default_filename)
                if os.path.exists(full_path):
                    self.configfile = full_path
                    return full_path
                
        # if we reach here, self.configfile will be still None, no valid
        # config files were found, and so we raise an exception
        raise IOError('ERROR - Can not find ' + self.default_filename + \
                      ' in your environment')

    def load(self):
        if not self.configfile:
            self.validate_configfile()
        parser = SafeConfigParser()
        parser.read(self.configfile)
        for section in self.sections:
            self.config[section] = {}
            for name, value in parser.items(section):
                self.config[section][name] = value
        return self.config        

    def get_section_parameters(self, section):
        if section not in self.sections:
            raise IndexError('ERROR - ' + section + \
                             ' is not one of the available sections: ' + \
                             ', '.join(self.sections))
        if section not in self.config.keys():
            # perhaps the config hasn't been loaded yet
            self.load()

        return self.config[section]

class MTAClientConfig(DaemonConfig):
    @property
    def default_filename():
        return 'postman-mta.conf'

    @property
    def sections(self):
        return ['postman_server', ]


def get_config_parameters(section=None, configfile=None):
    """
    This function is kept here for backwards compatibility.
    By default all the daemon code will use this function to get
    configuration parameters from the default configuration file.
    """
    config = DaemonConfig(configfile)
    return config.get_section_parameters(section)

    
