# -*- coding: utf-8 -*-

"""
The mailjam project - daemon.py

This file is released under the BSD license, see LICENSE for
more information.

Francisco de Borja Lopez Rio - <borja@codigo23.net>
Soluciones Informaticas Codigo23 S.L.U. - http://codigo23.net
"""

import os, multiprocessing, xmlrpclib, time
from SimpleXMLRPCServer import SimpleXMLRPCServer
from unittest import TestCase

from mailjam.daemon import Mailjam, MailjamXMLRPC, MailjamDaemon
from mailjam.models import Member, MailingList
from mailjam.storage import JsonStorage as Storage


class TestMailjam(TestCase):
    """
    mailjam.daemon.Mailjam tests.

    Remember to call the .clear() method of mailjam after each test, so
    the temp storage files are deleted
    """
    def setUp(self):
        self.configfile = os.path.join(os.path.dirname(__file__), 'mailjam.conf')
        self.mailing_list = MailingList('test_list', 'test_list@example.com',
                                        members={}, configfile=self.configfile)
        self.member =  Member('test@example.com')

    def test___init__(self):
        mailjam = Mailjam(configfile=self.configfile)
        self.assertIsInstance(mailjam, Mailjam)
        self.assertEqual(mailjam.mailings, {})
        self.assertEqual(mailjam.mailings_addresses, [])
        self.assertIsInstance(mailjam.dbs, dict)
        self.assertTrue('mailings' in mailjam.dbs.keys())
        self.assertTrue('members' in mailjam.dbs.keys())
        self.assertIsInstance(mailjam.dbs['mailings'], Storage)
        self.assertIsInstance(mailjam.dbs['members'], Storage)

    def test_save(self):
        mailjam = Mailjam(configfile=self.configfile)
        self.assertFalse(mailjam.save())
        mailjam.add_mailing_list(self.mailing_list.info())
        self.assertTrue(mailjam.save())
        # FIXME: We have to test here that the generated json file
        # contains the data it should contain

        # Clear the files created by the tests
        mailjam.clear()

    def test_load(self):
        mailjam = Mailjam(configfile=self.configfile)
        self.assertFalse(mailjam.mailings)
        self.assertFalse(mailjam.mailings_addresses)
        self.assertFalse(mailjam.load())
        mailjam.add_mailing_list(self.mailing_list.info())
        self.assertTrue(mailjam.load())

        # Check that another mailjam instance is able to load the saved data
        mailjam_load = Mailjam(configfile=self.configfile)
        self.assertFalse(mailjam_load.mailings)
        self.assertFalse(mailjam_load.mailings_addresses)
        mailjam_load.load()
        self.assertTrue(mailjam_load.mailings)
        self.assertIsInstance(mailjam_load.mailings, dict)
        self.assertTrue(mailjam_load.mailings_addresses)
        self.assertIsInstance(mailjam_load.mailings_addresses, list)        

        # Clear the files created by the tests
        mailjam.clear()

    def test_clear(self):
        mailjam = Mailjam(configfile=self.configfile)
        self.assertFalse(mailjam.clear())
        mailjam.add_mailing_list(self.mailing_list.info())
        self.assertTrue(mailjam.clear())

    def test_add_mailing_list(self):
        mailjam = Mailjam(configfile=self.configfile)
        with self.assertRaises(TypeError):
            # test improper info values
            mailjam.add_mailing_list(['a list', 'is an', 'invalid parameter'])
            mailjam.add_mailing_list(self.mailing_list)
        with self.assertRaises(ValueError):
            #test incomplete/missing info values
            mailjam.add_mailing_list()
            mailjam.add_mailing_list({'name': 'missing info'})
            mailjam.add_mailing_list({'address': 'missing info'})
            mailjam.add_mailing_list({'name': 'missing info',
                                      'address': 'missing info'})
        # test mailing lists can be added
        self.assertTrue(mailjam.add_mailing_list(self.mailing_list.info()))
        self.assertTrue(mailjam.mailings)
        self.assertIsInstance(mailjam.mailings, dict)
        self.assertTrue(mailjam.mailings_addresses)
        self.assertIsInstance(mailjam.mailings_addresses, list)
        with self.assertRaises(IndexError):
            # test what happens when the mailing has been already added
            mailjam.add_mailing_list(self.mailing_list.info())

        # Clear the files created by the tests
        mailjam.clear()

    def test_add_mailing_member(self):
        mailjam = Mailjam(configfile=self.configfile)
        mailjam.add_mailing_list(self.mailing_list.info())
        with self.assertRaises(ValueError):
            # test what happens if we call the method without proper
            # parameters
            mailjam.add_mailing_member()
            mailjam.add_mailing_member(None, None)
            mailjam.add_mailing_member(None, 'test_list@example.net')
            mailjam.add_mailing_member('test@example.net', None)
            mailjam.add_mailing_member('test@example', 'test_list@example.net')
        with self.assertRaises(IndexError):
            # test if we try to add a member to a non-existing mailing list
            mailjam.add_mailing_member('test@example.net',
                                       'test_list_b@example.net')
        # Test adding a member
        self.assertTrue(mailjam.add_mailing_member('test@example.net',
                                                   self.mailing_list.address))
        # Test trying to re-add that user
        self.assertFalse(mailjam.add_mailing_member('test@example.net',
                                                    self.mailing_list.address))

        # Clear the files created by the tests
        mailjam.clear()


class TestMailjamDaemon(TestCase):
    """
    mailjam.daemon.MailjamDaemon tests.

    Remember to call the .clear() method of mailjam after each test, so
    the temp storage files are deleted
    """
    def setUp(self):
        self.configfile = os.path.join(os.path.dirname(__file__),
                                       'mailjam.conf')
        self.mailing_list = MailingList('test_xmlrpc',
                                        'test_xmlrpc@example.com', members={},
                                        configfile=self.configfile)
        self.member =  Member('test@example.com')

    def test___init__(self):
        daemon = MailjamDaemon(self.configfile)
        self.assertIsInstance(daemon, MailjamDaemon)
        self.assertFalse(daemon.ready_to_serve)
        # FIXME: More tests should be added here once the configuration
        # file feature is added        

    def test_create_server(self):
        daemon = MailjamDaemon(self.configfile)
        daemon.port = 9001
        self.assertTrue(daemon.create_server())
        self.assertIsInstance(daemon.server, SimpleXMLRPCServer)
        self.assertFalse(daemon.ready_to_serve)
        
    def test_add_methods(self):
        daemon = MailjamDaemon(self.configfile)
        daemon.port = 9002
        self.assertTrue(daemon.add_methods())
        self.assertTrue(daemon.ready_to_serve)

        daemon = MailjamDaemon(self.configfile)
        daemon.port = 9003        
        daemon.create_server()
        self.assertTrue(daemon.add_methods())
        self.assertTrue(daemon.ready_to_serve)
        
    def test_run(self):
        daemon = MailjamDaemon(self.configfile)
        daemon.port = 9004
        # start the daemon in another process, so we can start communicating
        # with itjobs = []
        p = multiprocessing.Process(target=daemon.run)
        p.start()

        # Add a delay here to allow the server to be fully started before
        # starting to send requests from the client
        time.sleep(2)
        
        # FIXME: Hardcoded url here, should be picked from a config file        
        client = xmlrpclib.ServerProxy('http://localhost:9004')

        # Check that we can perform an XMLRPC call and that the list of
        # available public methods contains the list of methods we have
        # defined in our base XMLRPC class
        set_class_methods = set(MailjamXMLRPC()._listMethods())
        set_xmlrpc_methods = set(client.system.listMethods())
        self.assertTrue(set_class_methods.issubset(set_xmlrpc_methods))
        
        # Stop the server
        p.terminate()
        
        
