import unittest
import os

from pyenvjasmine.runner import Runner


class RunnerTests(unittest.TestCase):
    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_runner_defaults(self):
        """
        Test the runner, using default values (which wil run the demo specs)
        """
        jstests = Runner()
        output = jstests.run()
        # output is the resulting stdout from a subprocess call, which is a
        # bytes object, hence the b'' for the checks
        self.assertTrue(b'Failed: 0' in output)
        self.assertTrue(b'Passed: 5' in output)

    def test_runner_params(self):
        """
        Test the runner, giving it some parameters
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        conf_file = os.path.join(sample, 'configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            rootdir=envjasmine_dir,
            testdir=sample,
            configfile=conf_file,
        )
        output = jstests.run(spec='tests/specs/test_demo.spec.js')
        lines = output.splitlines()
        # output is the resulting stdout from a subprocess call, which is a
        # bytes object, hence the b'' in the endswith/startswith calls
        self.assertTrue(lines[0].endswith(b'specs/test_demo.spec.js'),
            "wrong output: %s" % str(output))
        self.assertTrue(lines[1].startswith(b'[  Envjs/1.6 (Rhino;'),
            "wrong output: %s" % str(output))
        self.assertTrue(b'Passed: 4' in lines)
        self.assertTrue(b'Failed: 0' in lines)
        self.assertTrue(b'Total : 4' in lines)

    def test_write_browser_htmlfile_markup_is_correct(self):
        """
        Test the created markup
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            rootdir=envjasmine_dir,
            testdir=sample,
            browser_configfile=browser_conf_file
        )
        expected = jstests.create_testRunnerHtml()
        jstests.run(spec='tests/specs/test_demo.spec.js')
        with open("browser.runner.html",'r') as file:
            actual = file.read()
            self.assertEqual(expected, actual)

    def test_runner_with_browser_configfile(self):
        """
        Test the runner, giving it some parameters incl the browser config file
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            rootdir=envjasmine_dir,
            testdir=sample,
            browser_configfile=browser_conf_file
        )
        jstests.run(spec='tests/specs/test_demo.spec.js')

    def test_get_environment(self):
        """
        Testing the OS specific code
        Could not figure out how to do this using mock,
        so monkey patching the old way.
        """
        from pyenvjasmine.runner import get_environment
        import os
        old_os_name = os.name
        try:
            os.name = 'nt'
            res = get_environment()
            self.assertEqual(res, '--environment=WIN')
            os.name = 'posix'
            res = get_environment()
            self.assertEqual(res, '--environment=UNIX')
        finally:
            os.name = old_os_name

    def test_not_capture_output(self):
        """
        Running a test without capturing output
        """
        jstests = Runner()
        output = jstests.run(capture_output=False)
        self.assertEqual(output, None)
