# -*- coding: utf-8 -*-

import os
from tools import validate_email_address
from storage import JsonStorage
import config

class Member():

    """
    Class that defines a mailing list member
    """

    def __init__(self, address):
        self.address = self._validate_address(address)
        
    def __repr__(self):
        return "<Member '%s'>" % self.address

    def __str__(self):
        return self.address

    def _validate_address(self, address):
        if not validate_email_address(address):
            raise ValueError(address, ' is not a valid email address')
        return address


class MailingList():

    """
    Class that defines a mailing list
    """

    def __init__(self, name, address, members={}, config={}, storage=None):
        self.name = name
        self.address = address
        self.members = members
        self.config = config
        self.storage = JsonStorage(os.path.join(config['storage'],
                                                self.address))

    def __repr__(self):
        return "<MailingList '%s'>" % self.address

    def __str__(self):
        return self.address

    def _validate_config(self):
        if not 'storage' in self.config.keys():
            self.config['storage'] = os.path.join(config.storage_path,
                                                  self.address)
        if not 'archive' in self.config.keys():
            self.config['archive'] = os.path.join(self.config['storage'],
                                                  'archive')
        if not 'private' in self.config.keys():
            self.config['private'] = config.private
        
    def _validate_member_object(self, member):
        if not isinstance(member, Member):
            raise TypeError(member, ' is not a valid Member instance')
        return member

    def _validate_member(self, member):
        member = self._validate_member_object(member)
        return member.address in self.members_addresses()

    def _validate_member_by_address(self, address):
        if not validate_email_address(address):
            raise ValueError(address, ' is not a valid email address')
        return address in self.members_addresses()

    def members_addresses(self):
        return self.members.keys()

    def add_member(self, member):
        member = self._validate_member_object(member)
        if self._validate_member(member):
            return False
        self.members[member.address] = member
        return True

    def add_member_by_address(self, address):
        if self._validate_member_by_address(address):
            return False
        member = Member(address)
        self.members[address] = member
        return True
                
    def delete_member(self, member):
        member = self._validate_member_object(member)
        if not self._validate_member(member):
            return False
        del self.members[member.address]
        return True

    def load(self):
        if self.storage.exists():
            data = self.storage.read()
            self.name = data.name
            self.address = data.address
            self.members = data.members
    
    def save(self):
        self.storage.write(self)
