import os

from pyenvjasmine.runner import Runner, get_environment


class TestsRunnerRhino(object):

    """
    Run the full tests using the old rhino+jasmine1 env
    """

    def test_runner_defaults(self):
        """
        Test the runner, using default values (which wil run the demo specs)
        """
        jstests = Runner(testing_enviroment='rhino')
        success, stdout = jstests.run()
        assert success
        assert 'Failed: 0' in stdout
        assert 'Passed: 5' in stdout

    def test_runner_params(self):
        """
        Test the runner, giving it some parameters
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        conf_file = os.path.join(sample, 'configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            testing_enviroment='rhino',
            rootdir=envjasmine_dir,
            testdir=sample,
            configfile=conf_file,
        )
        success, stdout = jstests.run(spec='tests/specs/test_demo.spec.js')
        lines = stdout.splitlines()
        assert lines[0].endswith('specs/test_demo.spec.js')
        assert lines[1].startswith('[  Envjs/1.6 (Rhino;')
        assert 'Passed: 4' in lines
        assert 'Failed: 0' in lines
        assert 'Total : 4' in lines

    def test_write_browser_htmlfile_markup_is_correct(self):
        """
        Test the created markup
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            testing_enviroment='rhino',
            rootdir=envjasmine_dir,
            testdir=sample,
            browser_configfile=browser_conf_file
        )
        expected = jstests.create_testRunnerHtml()
        success, stdout = jstests.run(spec='tests/specs/test_demo.spec.js')
        assert not success
        assert 'Failed: 2' in stdout
        with open("browser.runner.html",'r') as file:
            actual = file.read()
            assert expected == actual

    def test_runner_with_browser_configfile(self):
        """
        Test the runner, giving it some parameters incl the browser config file
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            testing_enviroment='rhino',
            rootdir=envjasmine_dir,
            testdir=sample,
            browser_configfile=browser_conf_file
        )
        success, stdout = jstests.run(spec='tests/specs/test_demo.spec.js')
        assert not success
        assert 'Failed: 2' in stdout

    def test_get_environment(self):
        """
        Testing the OS specific code
        Could not figure out how to do this using mock,
        so monkey patching the old way.
        """
        old_os_name = os.name
        try:
            os.name = 'nt'
            res = get_environment()
            assert res == '--environment=WIN'
            os.name = 'posix'
            res = get_environment()
            assert res == '--environment=UNIX'
        finally:
            os.name = old_os_name

    def test_did_test_pass(self):
        # there is some coverage done with the previous tests,
        # but we want to test also the case when a test failed
        # and it does not appear in the "Failed:" report
        jstests = Runner(testing_enviroment='rhino')
        success = jstests.did_test_pass('')
        assert not success
        success = jstests.did_test_pass('some random data '*50)
        assert not success
        success = jstests.did_test_pass('some data FAILED some more data')
        assert not success
        success = jstests.did_test_pass('some data FAILEDsome more data')
        assert not success
        success = jstests.did_test_pass('Failed: 0')
        assert success
        success = jstests.did_test_pass('Failed: 0 FAILED')
        assert not success
        success = jstests.did_test_pass('FAILEDFailed: 0')
        assert not success
        success = jstests.did_test_pass('Failed: 0FAILED')
        assert not success
        success = jstests.did_test_pass('Failed: 1')
        assert not success
        success = jstests.did_test_pass('Failed: -11')
        assert not success
        success = jstests.did_test_pass('Failed: something-not-a-number')
        assert not success

class TestsRunnerPhantomjs(object):

    """
    Run the full tests using the phantom+jasmine3 env
    """

    def test_write_browser_htmlfile_markup_is_correct(self):
        """
        Test the created markup
        The config file doesn't contain the mock files so jasmine tests
        are expected to fail
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile_no_mocks.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            testing_enviroment='phantomjs',
            rootdir=envjasmine_dir,
            testdir=sample,
            browser_configfile=browser_conf_file
        )
        expected = jstests.create_testRunnerHtml()
        jstests.write_browser_htmlfile()
        success, stdout = jstests.run()
        assert not success
        assert '2 test(s) FAILED:' in stdout
        with open("browser.runner.html",'r') as file:
            actual = file.read()
            assert expected == actual

    def test_runner_with_browser_configfile(self):
        """
        Test the runner, giving it some parameters incl the browser config file
        The config file contains the mock files so jasmine tests are expected
        to pass
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            testing_enviroment='phantomjs',
            rootdir=envjasmine_dir,
            testdir=sample,
            browser_configfile=browser_conf_file
        )
        success, stdout = jstests.run()
        assert success
        assert '0 failures' in stdout

    def test_did_test_pass(self):
        # there is some coverage done with the previous tests,
        # but we want to test also the case when a test failed
        # and it does not appear in the "Failed:" report
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile.js')

        jstests = Runner(
            testing_enviroment='phantomjs',
            browser_configfile=browser_conf_file
        )
        success = jstests.did_test_pass('')
        assert not success
        success = jstests.did_test_pass('some random data '*50)
        assert not success
        success = jstests.did_test_pass('some data FAILED some more data')
        assert not success
        success = jstests.did_test_pass('some data FAILEDsome more data')
        assert not success
        success = jstests.did_test_pass('0 failures')
        assert success
        success = jstests.did_test_pass('0 failures FAILED')
        assert not success
        success = jstests.did_test_pass('0Failed: failures')
        assert not success
        success = jstests.did_test_pass('1 test(s) FAILED:')
        assert not success
        success = jstests.did_test_pass('-11 test(s) FAILED:')
        assert not success
        success = jstests.did_test_pass('something-not-a-number test(s) FAILED:')
        assert not success
