# -*- coding: utf-8 -*-

import os
from unittest import TestCase
from mailjam.models import Member, MailingList


class TestMember(TestCase):
    def setUp(self):
        self.member = Member('test@example.com')

    def test__validate_address(self):
        self.assertEqual(self.member._validate_address(self.member.address),
                         self.member.address)
        with self.assertRaises(ValueError):
            self.member._validate_address('notavalidemail')

    
class TestMailingList(TestCase):
    def setUp(self):
        configfile = os.path.join(os.path.dirname(__file__), 'mailjam.conf')
        self.mailing_list = MailingList('test_list', 'test_list@example.com',
                                        members={}, configfile=configfile)
        self.member =  Member('test@example.com')

    def test__validate_member_object(self):
        self.assertEqual(self.mailing_list._validate_member_object(self.member),
                         self.member)
        with self.assertRaises(TypeError):
            self.mailing_list._validate_member_object(None)
            self.mailing_list._validate_member_object('Not a member object')
            self.mailing_list._validate_member_object(self.mailing_list)

    def test__validate_member(self):
        # At first the member object is not a member of the list
        self.assertFalse(self.mailing_list._validate_member(self.member))
        self.mailing_list.add_member(self.member)
        self.assertTrue(self.mailing_list._validate_member(self.member))
        with self.assertRaises(TypeError):
            self.mailing_list._validate_member(None)
            self.mailing_list._validate_member('Not a member object')
            self.mailing_list._validate_member(self.mailing_list)
        
    def test__validate_member_by_address(self):
        self.assertFalse(self.mailing_list._validate_member_by_address(self.member.address))
        self.mailing_list.add_member(self.member)
        self.assertTrue(self.mailing_list._validate_member_by_address(self.member.address))
        with self.assertRaises(ValueError):
            self.mailing_list._validate_member_by_address(self.member)
            self.mailing_list._validate_member_by_address(None)            
            self.mailing_list._validate_member_by_address('Not a member object')
            self.mailing_list._validate_member_by_address(self.mailing_list)

    def test_members_addresses(self):
        self.assertEqual(self.mailing_list.members_addresses(), [])
        self.mailing_list.add_member(self.member)
        self.assertTrue(self.member.address in \
                        self.mailing_list.members_addresses())

    def test_add_member(self):
        self.assertTrue(self.mailing_list.add_member(self.member))
        self.assertTrue(self.member.address in \
                        self.mailing_list.members_addresses())
        # check what happens if the member is already there:
        self.assertFalse(self.mailing_list.add_member(self.member))
        # check what happens if we try to add something that is not
        # a valid Member instance
        with self.assertRaises(TypeError):
            self.mailing_list.add_member(None)
            self.mailing_list.add_member('Not a member object')
            self.mailing_list.add_member(self.mailing_list)

    def test_add_member_by_address(self):
        self.assertTrue(self.mailing_list.add_member_by_address(self.member.address))
        self.assertTrue(self.member.address in \
                        self.mailing_list.members_addresses())
        # check what happens if the member is already there:
        self.assertFalse(self.mailing_list.add_member_by_address(self.member.address))
        # check what happens if we try to add something that is not
        # a valid Member instance
        with self.assertRaises(ValueError):
            self.mailing_list.add_member_by_address(self.member)
            self.mailing_list.add_member_by_address(None)            
            self.mailing_list.add_member_by_address('Not a member object')
            self.mailing_list.add_member_by_address(self.mailing_list)

    def test_delete_member(self):
        self.assertFalse(self.mailing_list.delete_member(self.member))
        self.mailing_list.add_member(self.member)
        self.assertTrue(self.member.address in \
                        self.mailing_list.members_addresses())
        self.assertTrue(self.mailing_list.delete_member(self.member))
        self.assertFalse(self.member.address in \
                         self.mailing_list.members_addresses())
        with self.assertRaises(TypeError):
            self.mailing_list.delete_member(None)
            self.mailing_list.delete_member('Not a member object')
            self.mailing_list.delete_member(self.mailing_list)
