import os
import pytest

from pyenvjasmine.runner import Runner, get_environment


class TestsRunner(object):

    def test_runner_defaults(self):
        """
        Test the runner, using default values (which wil run the demo specs)
        """
        jstests = Runner()
        success, stdout = jstests.run()
        assert success
        assert 'Failed: 0' in stdout
        assert 'Passed: 5' in stdout

    def test_runner_params(self):
        """
        Test the runner, giving it some parameters
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        conf_file = os.path.join(sample, 'configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            rootdir=envjasmine_dir,
            testdir=sample,
            configfile=conf_file,
        )
        success, stdout = jstests.run(spec='tests/specs/test_demo.spec.js')
        lines = stdout.splitlines()
        assert lines[0].endswith('specs/test_demo.spec.js')
        assert lines[1].startswith('[  Envjs/1.6 (Rhino;')
        assert 'Passed: 4' in lines
        assert 'Failed: 0' in lines
        assert 'Total : 4' in lines

    def test_write_browser_htmlfile_markup_is_correct(self):
        """
        Test the created markup
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            rootdir=envjasmine_dir,
            testdir=sample,
            browser_configfile=browser_conf_file
        )
        expected = jstests.create_testRunnerHtml()
        jstests.run(spec='tests/specs/test_demo.spec.js')
        with open("browser.runner.html",'r') as file:
            actual = file.read()
            assert expected == actual

    def test_runner_with_browser_configfile(self):
        """
        Test the runner, giving it some parameters incl the browser config file
        """
        here = os.path.dirname(__file__)
        sample = os.path.join(here, 'sample')
        browser_conf_file = os.path.join(sample, 'browser.configfile.js')
        envjasmine_dir = os.path.join(os.path.dirname(here), 'envjasmine')
        jstests = Runner(
            rootdir=envjasmine_dir,
            testdir=sample,
            browser_configfile=browser_conf_file
        )
        success, stdout = jstests.run(spec='tests/specs/test_demo.spec.js')
        assert not success

    def test_get_environment(self):
        """
        Testing the OS specific code
        Could not figure out how to do this using mock,
        so monkey patching the old way.
        """
        old_os_name = os.name
        try:
            os.name = 'nt'
            res = get_environment()
            assert res == '--environment=WIN'
            os.name = 'posix'
            res = get_environment()
            assert res == '--environment=UNIX'
        finally:
            os.name = old_os_name
